let curSE = getCurSE() // 存储当前搜索引擎
let keyword = "" // 存储当前搜索关键字

/**
 * 搜索引擎映射
 * @typedef {Object} SearchEngine
 * @property {string} name - 搜索引擎的名称
 * @property {string} url - 搜索引擎的查询 URL 模板
 * @property {string} query - 查询参数的键名
 * @property {string} icon - 搜索引擎的图标 URL
 */

/**
 * @type {Object.<string, SearchEngine>}
 */
const SE = {
  baidu: {
    name: "百度",
    url: "https://www.baidu.com/s?wd=",
    query: "wd",
    icon: "https://www.baidu.com/favicon.ico"
  },
  google: {
    name: "谷歌",
    url: "https://www.google.com.hk/search?q=",
    query: "q",
    icon: "https://www.google.com.hk/favicon.ico"
  },
  bing: {
    name: "必应",
    url: "https://cn.bing.com/search?q=",
    query: "q",
    icon: "https://cn.bing.com/sa/simg/bing_p_rr_teal_min.ico"
  },
  sougou: {
    name: "搜狗",
    url: "https://www.sogou.com/web?query=",
    query: "query",
    icon: "https://sogou.com/images/logo/new/favicon.ico?nv=1&v=3"
  },
  360: {
    name: "360",
    url: "https://www.so.com/s?q=",
    query: "q",
    icon: "https://www.so.com/favicon.ico"
  }
}

// 获取当前搜索引擎
function getCurSE() {
  let curSE = ""
  const host = window.location.host
  if (host === "www.baidu.com") {
    curSE = "baidu"
  } else if (host.includes("google")) {
    curSE = "google"
  } else if (host.includes("bing.com")) {
    curSE = "bing"
  } else if (host === "www.sogou.com") {
    curSE = "sougou"
  } else if (host === "www.so.com") {
    curSE = "360"
  }
  return curSE
}
// 获取搜索关键字
function getSearchKeyword(curSE) {
  const search = window.location.search
  const query = new URLSearchParams(search)
  if (!query) {
    return
  }
  return query.get(SE[curSE]?.query)
}
// 生成搜索引擎a标签
function createSEItem(name, url, key) {
  const item = document.createElement("div")
  item.classList.add("ovo-search-engine-nav-item")
  item.innerHTML = `<img src="${SE[key].icon}" alt="${name}" >`

  // a标签
  const a = document.createElement("a")
  a.href = url
  a.innerHTML = `${name}`
  // 当前tab打开
  a.target = "_self" // bing打开新tab 不然广告过滤有问题
  a.setAttribute("data-key", key)

  item.append(a)

  return item
}
// 获取搜索引擎导航
function generateSearchEngineNav(curSE) {
  const searchEngineNav = document.createElement("div")
  searchEngineNav.classList.add("ovo-search-engine-nav")
  keyword = getSearchKeyword(curSE)
  if (!keyword) return
  const navs = Object.keys(SE).map((key) => {
    const se = SE[key]
    return createSEItem(se.name, se.url + keyword, key)
  })
  searchEngineNav.append(...navs)
  document.body.appendChild(searchEngineNav)
}

// 更新搜索引擎导航
function updateSearchEngineNav(curSE) {
  const searchEngineNav = document.querySelector(".ovo-search-engine-nav")
  keyword = getSearchKeyword(curSE)
  if (!keyword) return
  if (searchEngineNav) {
    const navs = searchEngineNav.querySelectorAll(".ovo-search-engine-nav-item")
    navs.forEach((nav) => {
      const aDom = nav.querySelector("a")
      const key = aDom.getAttribute("data-key")
      const url = SE[key].url + keyword
      aDom.href = url
    })
  } else {
    generateSearchEngineNav(curSE)
  }
}

const debounceUpdateSearchEngineNav = debounce(updateSearchEngineNav, 1000)

// 监听 DOM 变化回调
function seMutationObserverCallback() {
  requestAnimationFrame(() => {
    curSE = getCurSE()
    curSE && debounceUpdateSearchEngineNav(curSE)
  })
}

// 监听 DOM 变化
function setupChangeSEMutationObserver() {
  const observer = new MutationObserver(seMutationObserverCallback)
  const config = { childList: true, subtree: true } // 监听子节点变化 和 所有后代节点变化
  observer.observe(document.body, config)
}

// 防抖
function debounce(fn, delay) {
  let timer = null
  return function () {
    clearTimeout(timer)
    timer = setTimeout(() => {
      fn.apply(this, arguments)
    }, delay)
  }
}

window.chrome.storage.sync.get(["OVO_CHANGE_SEARCH_ENGINE"], (result) => {
  if (result.OVO_CHANGE_SEARCH_ENGINE) {
    generateSearchEngineNav(curSE)
    setupChangeSEMutationObserver()
  }
})
